#ifndef V3D_BASICS_VECTOR_TPL_H_INCLUDED
#define V3D_BASICS_VECTOR_TPL_H_INCLUDED

#include "Maths.h"

namespace V3D {


template <typename REAL>
class Vector3TplBase
{
public:
	REAL   x, y, z;
public:
	inline Vector3TplBase() {}
//	template <typename OTHERREAL>
//		inline Vector3TplBase(const Vector3TplBase<OTHERREAL>& other):x(other.x), y(other.y), z(other.z) {}
	
	inline Vector3TplBase( REAL newx, REAL newy, REAL newz):x(newx), y(newy), z(newz) {}
	
	
	// Conversion d'un type de vecteur3 a un autre. Generera eventuellement des warnings (double->float)
	template <typename OTHERREAL>
		inline Vector3TplBase& Set( const Vector3TplBase<OTHERREAL>& other)
		{ x = other.x; y = other.y; z = other.z; return *this;}

	// Conversion d'un type de vecteur3 a un autre. Aucun warning
	template <typename OTHERREAL>
		inline Vector3TplBase& SetUnsafe( const Vector3TplBase<OTHERREAL>& other)
		{ x = static_cast<REAL>(other.x); y = static_cast<REAL>(other.y); z = static_cast<REAL>(other.z); return *this;}

	// Conversion d'un type de vecteur3 a un autre. Aucun warning
	template <typename OTHERREAL>
		inline static Vector3TplBase ConvertFrom( const Vector3TplBase<OTHERREAL>& other)
		{ return Vector3TplBase( static_cast<REAL>(other.x), static_cast<REAL>(other.y), static_cast<REAL>(other.z) );}

	
	inline Vector3TplBase& Set(REAL newx, REAL newy, REAL newz);
	
	inline REAL operator [] (int32 i) const { return (&x)[i];}
	inline REAL& operator [] (int32 i) { return (&x)[i];}
	
	inline void Normalize();
	inline Vector3TplBase Normalized() const;

	inline Vector3TplBase RotateX( float fAngle ) const;
	inline Vector3TplBase RotateY( float fAngle ) const;
	inline Vector3TplBase RotateZ( float fAngle ) const;


	inline REAL GetSquareLength() const;
	inline REAL GetLength() const;
	
	// Produit scalaire
	inline REAL operator * (const Vector3TplBase<REAL>& vec) const;

	inline Vector3TplBase operator - () const;

	inline Vector3TplBase& operator *= (REAL fact);
	inline Vector3TplBase& operator /= (REAL fact);
	inline Vector3TplBase& operator += (const Vector3TplBase& vec);
	inline Vector3TplBase& operator -= (const Vector3TplBase& vec);
	inline Vector3TplBase& operator ^= (const Vector3TplBase& vec);

	inline bool operator == (const Vector3TplBase& vec) const;
	inline bool IsEqual( const Vector3TplBase<REAL>& vtx, float fEpsilon =  1E-6f ) const;

	inline operator const REAL*() const { return static_cast<const REAL*>(&x); }
	inline operator REAL*() { return static_cast<REAL*>(&x); }


};




template <class REAL>
class Vector3Tpl : public Vector3TplBase<REAL>
{
public:
	inline Vector3Tpl() {}
	inline Vector3Tpl( REAL newx, REAL newy, REAL newz): Vector3TplBase<REAL>(newx, newy, newz) {}
	inline Vector3Tpl( const Vector3TplBase<REAL>& ori): Vector3TplBase<REAL>(ori.x, ori.y, ori.z) {}
};



template <>
class Vector3Tpl<double> : public Vector3TplBase<double>
{
public:
	inline Vector3Tpl() {}
	inline Vector3Tpl( double newx, double newy, double newz): Vector3TplBase<double>(newx, newy, newz) {}

	// Conversion depuis vecteur de floats
	explicit inline Vector3Tpl(const Vector3TplBase<float>& other) : Vector3TplBase<double>(other.x, other.y, other.z) {}

	// Constructeur de copie
	inline Vector3Tpl(const Vector3Tpl& other) : Vector3TplBase<double>(other.x, other.y, other.z) {}
	inline Vector3Tpl( const Vector3TplBase<double>& ori): Vector3TplBase<double>(ori.x, ori.y, ori.z) {}


};






////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////

// Vecteur 4D, classe tres simple car peu utilisee
template <typename REAL>
class Vector4Tpl
{
public:
	REAL x, y, z, w;

public:
	inline Vector4Tpl() {}
	inline Vector4Tpl(REAL _x, REAL _y, REAL _z, REAL _w) : x(_x), y(_y), z(_z), w(_w) {}
	inline Vector4Tpl(const Vector4Tpl& vec) : x(vec.x), y(vec.y), z(vec.z), w(vec.w) {}
	explicit inline Vector4Tpl(const Vector3TplBase<REAL>& vec) : x(vec.x), y(vec.y), z(vec.z), w(1.f) {}

	inline Vector4Tpl& Set(REAL _x, REAL _y, REAL _z, REAL _w) {x=_x; y=_y; z=_z; w=_w; return *this;}

	template <typename OTHERREAL>
		inline Vector4Tpl& Set( const Vector3TplBase<OTHERREAL>& other, OTHERREAL otherW=0)
		{ x = static_cast<REAL>(other.x); y = static_cast<REAL>(other.y); z = static_cast<REAL>(other.z); w = otherW; return *this;}

	template <typename OTHERREAL>
		inline Vector4Tpl& Set( const Vector4Tpl<OTHERREAL>& other)
		{ x = static_cast<REAL>(other.x); y = static_cast<REAL>(other.y); z = static_cast<REAL>(other.z); w=static_cast<REAL>(other.w); return *this;}

	inline REAL operator * (const Vector4Tpl& vc) {return (x * vc.x) + (y * vc.y) +(z * vc.z) +(w * vc.w);}

	inline operator const REAL*() const { return static_cast<const REAL*>(&x); }
	inline operator REAL*() { return static_cast<REAL*>(&x); }

};

////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////

	// Definition des methodes inline de Vector3TplBase


template <typename REAL>
inline Vector3TplBase<REAL>& Vector3TplBase<REAL>::Set(REAL newx, REAL newy, REAL newz)
{
	x = newx; y = newy; z = newz;
	return *this;
}


template <typename REAL>
inline REAL Vector3TplBase<REAL>::GetSquareLength() const
{
	return x*x + y*y + z*z;
}

template <typename REAL>
inline REAL Vector3TplBase<REAL>::GetLength() const
{
	return static_cast<REAL> (sqrt(Vector3TplBase<REAL>::GetSquareLength()));
}


template <typename REAL>
inline void Vector3TplBase<REAL>::Normalize()
{
	REAL divLength = 1.f / GetLength();
	x *= divLength;
	y *= divLength;
	z *= divLength;
}

template <typename REAL>
inline Vector3TplBase<REAL> Vector3TplBase<REAL>::Normalized() const
{
	REAL rDivLength = 1.f / GetLength();
	return Vector3TplBase<REAL>( x * rDivLength, y * rDivLength, z * rDivLength);
}


template <typename REAL>
inline Vector3TplBase<REAL> Vector3TplBase<REAL>::RotateX( float fAngle ) const
{
	float fCosAng = Maths::Cos( fAngle );
	float fSinAng = Maths::Sin( fAngle );

	return Vector3TplBase(              x,
	                       fCosAng  * y + fSinAng * z,
	                       -fSinAng * y + fCosAng * z );
}

template <typename REAL>
inline Vector3TplBase<REAL> Vector3TplBase<REAL>::RotateY( float fAngle ) const
{
	float fCosAng = Maths::Cos( fAngle );
	float fSinAng = Maths::Sin( fAngle );

	return Vector3TplBase( fCosAng * x - fSinAng * z,
	                                      y,
	                       fSinAng * x + fCosAng * z );
}

template <typename REAL>
inline Vector3TplBase<REAL> Vector3TplBase<REAL>::RotateZ( float fAngle ) const
{
	float fCosAng = Maths::Cos( fAngle );
	float fSinAng = Maths::Sin( fAngle );

	return Vector3TplBase( fCosAng  * x + fSinAng * y,
	                       -fSinAng * x + fCosAng * y,
	                                      z           );
}




template <typename REAL>
inline Vector3TplBase<REAL>& Vector3TplBase<REAL>::operator *= (REAL fact)
{
	x *= fact;
	y *= fact;
	z *= fact;
	return *this;
}


template <typename REAL>
inline Vector3TplBase<REAL>& Vector3TplBase<REAL>::operator /= (REAL div)
{
	REAL inv = 1 / div;
	x *= inv;
	y *= inv;
	z *= inv;
	return *this;
}


template <typename REAL>
inline Vector3TplBase<REAL>& Vector3TplBase<REAL>::operator += (const Vector3TplBase<REAL>& vec)
{
	x += vec.x;
	y += vec.y;
	z += vec.z;
	return *this;
}

template <typename REAL>
inline Vector3TplBase<REAL>& Vector3TplBase<REAL>::operator -= (const Vector3TplBase<REAL>& vec)
{
	x -= vec.x;
	y -= vec.y;
	z -= vec.z;
	return *this;
}

template <typename REAL>
inline Vector3TplBase<REAL>& Vector3TplBase<REAL>::operator ^= (const Vector3TplBase<REAL>& vec)
{
	REAL vx = vec.x;
	REAL vy = vec.y;
	REAL vz = vec.z;
	REAL rCrossX = y * vz - z * vy;
	REAL rCrossY = z * vx - x * vz;
	REAL rCrossZ = x * vy - y * vx;
	x = rCrossX;
	y = rCrossY;
	z = rCrossZ;
	return *this;
}

// Produit scalaire
template <typename REAL>
inline REAL Vector3TplBase<REAL>::operator * (const Vector3TplBase<REAL>& vc) const
{
	return x * vc.x + y * vc.y + z * vc.z;
}


template <typename REAL>
inline Vector3TplBase<REAL> Vector3TplBase<REAL>::operator - () const
{
	return Vector3TplBase<REAL>( -x, -y, -z);
}


template <typename REAL>
inline bool Vector3TplBase<REAL>::operator == (const Vector3TplBase<REAL>& vc) const
{
	return (x == vc.x) && (y == vc.y) && (z == vc.z);
}

template <typename REAL>
inline bool Vector3TplBase<REAL>::IsEqual( const Vector3TplBase<REAL>& vtx, float fEpsilon /* =  1E-6f */ ) const
{
	float fSqrEpsilon = fEpsilon * fEpsilon;
	if( (x-vtx.x) * (x-vtx.x) > fSqrEpsilon ) return false;
	if( (y-vtx.y) * (y-vtx.y) > fSqrEpsilon ) return false;
	if( (z-vtx.z) * (z-vtx.z) > fSqrEpsilon ) return false;
	return true;
}





// Methodes utilitaires

// Operations de base entre types mixtes (Vector3TplBase<float> et Vector3TplBase<double>)


inline double operator * (const Vector3TplBase<float>& vc1, const Vector3TplBase<double>& vc2)
{
	return ( vc1.x * vc2.x + vc1.y * vc2.y + vc1.z * vc2.z);
}

inline double operator * (const Vector3TplBase<double>& vc1, const Vector3TplBase<float>& vc2)
{
	return ( vc1.x * vc2.x + vc1.y * vc2.y + vc1.z * vc2.z);
}

inline Vector3TplBase<double> operator + (const Vector3TplBase<double>& vc1, const Vector3TplBase<float>& vc2)
{
	return Vector3TplBase<double>( vc1.x + vc2.x, vc1.y + vc2.y, vc1.z + vc2.z);
}

inline Vector3TplBase<double> operator + (const Vector3TplBase<float>& vc1, const Vector3TplBase<double>& vc2)
{
	return Vector3TplBase<double>( vc1.x + vc2.x, vc1.y + vc2.y, vc1.z + vc2.z);
}

inline Vector3TplBase<double> operator - (const Vector3TplBase<double>& vc1, const Vector3TplBase<float>& vc2)
{
	return Vector3TplBase<double>( vc1.x - vc2.x, vc1.y - vc2.y, vc1.z - vc2.z);
}

inline Vector3TplBase<double> operator - (const Vector3TplBase<float>& vc1, const Vector3TplBase<double>& vc2)
{
	return Vector3TplBase<double>( vc1.x - vc2.x, vc1.y - vc2.y, vc1.z - vc2.z);
}





template <typename REAL>
inline Vector3TplBase<REAL> Normalized( const Vector3TplBase<REAL>& vc )
{
	return vc.Normalized();
}


template <typename REAL>
inline bool operator != (const Vector3TplBase<REAL>& vc1, const Vector3TplBase<REAL>& vc2)
{
	return !(vc1 == vc2);
}



template <typename REAL>
inline Vector3TplBase<REAL> operator ^ (const Vector3TplBase<REAL>& vc1, const Vector3TplBase<REAL>& vc2)
{
	Vector3TplBase<REAL> res = vc1;
	return res ^= vc2;
}


template <typename REAL>
inline Vector3TplBase<REAL> operator + (const Vector3TplBase<REAL>& vc1, const Vector3TplBase<REAL>& vc2)
{
	Vector3TplBase<REAL> res = vc1;
	return res += vc2;
}


template <typename REAL>
inline Vector3TplBase<REAL> operator - (const Vector3TplBase<REAL>& vc1, const Vector3TplBase<REAL>& vc2)
{
	Vector3TplBase<REAL> res = vc1;
	return res -= vc2;
}


template <typename REAL>
inline Vector3TplBase<REAL> operator / (const Vector3TplBase<REAL>& vc, float f)
{
	Vector3TplBase<REAL> res = vc;
	return res /= f;
}

template <typename REAL>
inline Vector3TplBase<REAL> operator * (const Vector3TplBase<REAL>& vc, double f)
{
	Vector3TplBase<REAL> res = vc;
	return res *= REAL(f);
}

template <typename REAL>
inline Vector3TplBase<REAL> operator * (double f, const Vector3TplBase<REAL>& vc)
{
	Vector3TplBase<REAL> res = vc;
	return res *= REAL(f);
}



} // namespaces


#endif	// #ifndef V3D_BASICS_VECTOR_TPL_H_INCLUDED
